﻿using SWConfigDataClientLib;
using SWConfigDataClientLib.Data;
using SWConfigDataClientLib.Proxies.Admin;
using SWConfigDataClientLib.Proxies.Files;
using SWConfigDataClientLib.Proxies.PhoneClient;
using SWConfigDataClientLib.Proxies.UserPhoneBook;
using SWConfigDataClientLib.WSPhoneClientFacade;
using SWConfigDataSharedLib.Tracing;
using System;
using System.Collections.Generic;

namespace IpPbx.Samples.Shared
{
    /// <summary>
    /// Contains all methods which uses the CDSClient
    /// </summary>
    public class CDSLib : IDisposable
    {
        #region Tracing

        /// <summary>
        /// Init user defined trace module named "CDSlib" 
        /// See App.config or Web.config for trace module configuration
        /// </summary>
        private static readonly STraceInfo _log = new STraceInfo(
            new System.Diagnostics.TraceSwitch("CDSLib", "CDSlib", "2"), typeof(CDSLib));

        #endregion

        #region Fields

        /// <summary>
        /// The CDS Client LibManager is the central class of the CDS Client API.
        /// It contains methods to address the server and to authenticate users.
        /// You use LibManager to create instances of classes for retrieval and
        /// manipulation of IpPbx configuration items. The LibManager should be the
        /// first class to instantiate. It checks the configuration and initiates
        /// the tracing of the client application. 
        /// </summary>
        private LibManager _libManager;

        /// <summary>
        /// Web Service proxy for IpPbx clients. It contains specialized views
        /// and methods to be used from client application. A PhoneClientFacade
        /// instance is implicitly associated to the IpPbx user specified in
        /// the lib manager.
        /// </summary>
        private PhoneClientFacade _phoneClientFacade;

        /// <summary>
        /// Web service proxy for administration clients. It contains specialized
        /// views analogue to the views used in the IpPbx Administration.
        /// If your application does any administrative tasks, look at this object first.
        /// </summary>
        private AdminFacade _adminFacade;

        /// <summary>
        /// Web service proxy for managing all files stored in IpPbx's database.
        /// </summary>
        private FilesFacade _filesFacade;

        #endregion

        #region Properties

        /// <summary>
        /// Returns the name of the current user. A username is
        /// required for the UsernamePassword authentication mode.
        /// </summary>
        public string UserName
        {
            get
            {
                if (_libManager == null ||
                    string.IsNullOrWhiteSpace(_libManager.Username))
                {
                    return null;
                }

                return _libManager.Username;
            }
        }

        /// <summary>
        /// Returns the server address of the web service.        
        /// </summary>
        public string ServerName
        {
            get
            {
                if (_libManager == null ||
                    string.IsNullOrWhiteSpace(_libManager.WSBaseUrl))
                {
                    return null;
                }

                return _libManager.WSBaseUrl;
            }
        }

        /// <summary>
        /// Returns the current AuthenticationMode
        /// </summary>  
        public SProxyObject.AuthenticationMode AuthenticationMode
        {
            get
            {
                if (_libManager == null)
                {
                    return SProxyObject.AuthenticationMode.NoAuthentification;
                }

                return _libManager.AuthenticationMode;
            }
        }

        #endregion

        //------------------------------------------------------------------------------------

        #region CTors

        /// <summary>
        /// Creates a new CDS object (Windows Authentication)
        /// Assumes that the SwyxServer is running on the local machine (localhost)
        /// </summary>        
        public CDSLib()
            : this(String.Empty, String.Empty, String.Empty)
        {
            // Nothing else to do
        }

        /// <summary>
        /// Creates a new CDS object (Windows Authentication)
        /// </summary>     
        /// <param name="serverName">SwyxServer Address (FQDN or IP)</param>
        public CDSLib(string serverName)
            : this(String.Empty, String.Empty, serverName)
        {
            // Nothing else to do
        }

        /// <summary>
        /// Creates a new CDS object (UsernamePassword Authentication)
        /// Assumes that the SwyxServer is running on the local machine (localhost)
        /// </summary>
        /// <param name="userName">Username (SwyxIt Login)</param>
        /// <param name="password">Password (SwyxIt Login)</param>        
        public CDSLib(string userName, string password)
            : this(userName, password, String.Empty)
        {
            // Nothing else to do
        }

        /// <summary>
        /// Creates a new CDS object (UsernamePassword Authentication)
        /// </summary>
        /// <param name="userName">Username (SwyxIt Login)</param>
        /// <param name="password">Password (SwyxIt Login)</param>
        /// <param name="serverName">SwyxServer Address (FQDN or IP)</param>
        public CDSLib(string userName, string password, string serverName)
        {
            Init(userName, password, serverName);
        }

        #endregion

        #region Initialisation

        /// <summary>
        /// Initialisation
        /// </summary>
        /// <param name="userName">Username (SwyxIt Login)</param>
        /// <param name="password">Password (SwyxIt Login)</param>
        /// <param name="serverAddress">SwyxServer Address (FQDN or IP)</param>
        private void Init(string userName, string password, string serverAddress)
        {
            _log.TraceInfo("Init", "()");

            // Check if server address is empty
            if (string.IsNullOrWhiteSpace(serverAddress))
            {
                serverAddress = "127.0.0.1";
            }

            _log.TraceInfo("Init", "ServerAddress: {0}", serverAddress);

            // Set default authentication mode to trusted
            var authenticationMode = SProxyObject.AuthenticationMode.Trusted;
            if (serverAddress == "localhost" ||
                serverAddress == "127.0.0.1")
            {
                authenticationMode = SProxyObject.AuthenticationMode.TrustedLocal;
            }

            // If username and password are available the authentication
            // mode must be changed to UsernamePassword
            if (!string.IsNullOrWhiteSpace(userName) &&
                !string.IsNullOrWhiteSpace(password))
            {
                authenticationMode = SProxyObject.AuthenticationMode.UsernamePassword;
            }

            _log.TraceInfo("Init", "AuthenticationMode: {0}", authenticationMode);

            if (authenticationMode == SProxyObject.AuthenticationMode.UsernamePassword)
            {
                _log.TraceInfo("Init", "Username: {0}", userName);
            }

            // Create and configure new LibManager object
            _libManager = new LibManager()
            {
                BaseProtocol = SProxyObject.BaseProtocol.TCP,
                Username = userName,
                Password = password,
                WSBaseUrl = serverAddress,
                AuthenticationMode = authenticationMode
            };

            _log.TraceInfo("Init", "CDSLib initialisation completed");
        }

        #endregion

        #region IDisposable Members

        /// <summary>
        /// Performs application-defined tasks associated with freeing,
        /// releasing, or resetting unmanaged resources.
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Performs application-defined tasks associated with freeing,
        /// releasing, or resetting unmanaged resources.
        /// </summary>
        protected virtual void Dispose(bool disposing)
        {
            if (!disposing)
            {
                return;
            }

            // Cleanup
            if (_adminFacade != null)
            {
                _adminFacade.FreeForReuse();
                _adminFacade = null;
            }

            if (_filesFacade != null)
            {
                _filesFacade.FreeForReuse();
                _filesFacade = null;
            }

            if (_phoneClientFacade != null)
            {
                _phoneClientFacade.FreeForReuse();
                _phoneClientFacade = null;
            }

            if (_libManager != null)
            {
                _libManager.FreeForReuse();
                _libManager = null;
            }
        }

        #endregion

        #region Test Connection

        /// <summary>
        /// Check if a CDS connection can be established
        /// The use of this method is NOT mandatory.
        /// </summary>
        public bool TestConnection()
        {
            _log.TraceInfo("TestConnection", "()");

            return HasRole(BuildInRoles.UserSelf) || HasRole(BuildInRoles.Admin);
        }

        #endregion

        //------------------------------------------------------------------------------------
        // SAMPLE METHODS
        //------------------------------------------------------------------------------------

        #region Files (Get Skin List & Upload Skin)

        /// <summary>
        /// Return the path to the file in the IpPbx file cache (on local harddisk)
        /// </summary>
        public string GetFile(FileListEntry fileEntry)
        {
            // Ensure that a FilesFacade object is available
            if (_filesFacade == null)
            {
                _filesFacade = _libManager.GetFilesFacade();
            }

            return _filesFacade.GetIpPbxFile(fileEntry);
        }

        //------------------------------------------------------------------------------------

        /// <summary>
        /// Adds a skin file to the database into the global scope
        /// The skin will be available for every ippbx user
        /// </summary>
        /// <param name="filename">Name of the file in the database</param>
        /// <param name="filePath">Source path from where file should be uploaded</param>
        public void AddSkin(string filename, string filePath)
        {
            // Ensure that a FilesFacade object is available
            if (_filesFacade == null)
            {
                _filesFacade = _libManager.GetFilesFacade();
            }

            // Upload file to database
            _filesFacade.AddIpPbxFile(filename,
                                          FileScope.Global,
                                          PreDefinedFileCategory.Skins,
                                          0,
                                          false,
                                          false,
                                          "Uploaded with CDS Client SDK FileDemo App",
                                          false,
                                          false,
                                          "en-GB",
                                          filePath,
                                          true);
        }

        //------------------------------------------------------------------------------------

        /// <summary>
        /// Returns a file list with all global and system skins
        /// </summary>
        public FileListEntrySortableCollection GetSkins()
        {
            // Ensure that a FilesFacade object is available
            if (_filesFacade == null)
            {
                _filesFacade = _libManager.GetFilesFacade();
            }

            // Get skins from global scope
            var globalSkins = _filesFacade.GetFileList("%",
                FileScope.Global, PreDefinedFileCategory.Skins, 0, true).CreateSortableFilterCollection();

            // Get skins from system scope (SystemDefault)
            var systemSkins = _filesFacade.GetFileList("%",
                FileScope.SystemDefault, PreDefinedFileCategory.Skins, 0, true).CreateSortableFilterCollection();

            // Merge system and global skins into one list
            var allSkins = new FileListEntrySortableCollection();
            allSkins.AddRange(globalSkins);
            allSkins.AddRange(systemSkins);

            return allSkins;
        }

        #endregion

        #region User Roles

        /// <summary>
        /// Checks if the current user has the given role
        /// </summary>
        /// <param name="role">User Role (e.g. UserSelf or Admin)</param>
        public bool HasRole(BuildInRoles role)
        {
            _log.TraceInfo("HasRole", "Check for: {0}", role);

            // Ensure that a PhoneClientFacade object is available
            if (_phoneClientFacade == null)
            {
                _phoneClientFacade = _libManager.GetPhoneClientFacade();
            }

            // Check if user has the role
            if (_phoneClientFacade.HasRole(role))
            {
                // User has the role
                _log.TraceVerbose("HasRole", "Success");
                return true;
            }

            // User does NOT have the role
            _log.TraceVerbose("HasRole", "Failed");
            return false;
        }

        #endregion

        #region Users (Read only)

        /// <summary>
        /// Returns all users in one collection (READ ONLY)
        /// </summary>
        /// <param name="includeUserFileSize">If true the size of the user files in the database is calculated and added</param>
        public UserAdminView1EntrySortableCollection GetUserView(bool includeUserFileSize)
        {
            _log.TraceInfo("GetUserView", "()");

            // Ensure that a AdminFacade object is available
            if (_adminFacade == null)
            {
                _adminFacade = _libManager.GetAdminFacade();
            }

            try
            {
                return _adminFacade.GetUserAdminView1(includeUserFileSize,
                        out var userDeviceSmallMap).CreateSortableFilterCollection();
            }
            catch (Exception exception)
            {
                _log.TraceException(exception, true);
                throw;
            }
        }

        #endregion

        #region Users (Read & Write)

        /// <summary>
        /// Returns a collection of all ippbx users
        /// </summary>
        public SWConfigDataClientLib.Proxies.Users.UserPrimaryCollection GetUsers()
        {
            _log.TraceInfo("GetUsers", "()");

            // If only "%" (SQL Wildcard) is used for the
            // "nameFilter" parameter all users will be returned
            return GetUsers("%");
        }

        //------------------------------------------------------------------------------------

        /// <summary>
        /// Returns a collection of ippbx users filtered by name
        /// </summary>
        /// <param name="nameFilter">Name of the IpPbx user your are
        /// searching for (SQL Wildcards e.g. "%" can be used here)</param>
        public SWConfigDataClientLib.Proxies.Users.UserPrimaryCollection GetUsers(string nameFilter)
        {
            _log.TraceInfo("GetUsers", "Filter: {0}", nameFilter);

            // Get UserEnum object from LibManager
            var userEnum = _libManager.GetUserEnum();

            try
            {

                userEnum.ExecuteNameFilter(nameFilter, new OrderByList());
                return userEnum.PrimaryCollection;
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
            finally
            {
                // IMPORTANT
                // If you free resources here you will NOT be able
                // to save changes anymore. Every UserEntry has a
                // reference to his parent UserEnum to save changes.
                // If the Enum is released the reference will be broken.
                // For "read-only" purposes the AdminFacade should be
                // used because it is much faster and lightweight.

                /* Free resources
                userEnum?.FreeForReuse();
                */
            }
        }

        //------------------------------------------------------------------------------------

        public void DeleteUser(int userId)
        {
            _log.TraceInfo("UpdateUser", $"UserId: '{userId}'");

            // Get UserEnum object from LibManager
            var userEnum = _libManager.GetUserEnum();

            try
            {
                // Fill PrimaryCollection with all users
                userEnum.ExecuteIDFilter(userId);
                if (userEnum.PrimaryCollection.Count == 0)
                {
                    throw new ArgumentException($"A user with id {userId} could not be found.");
                }

                // Remove user from collection
                userEnum.PrimaryCollection.RemoveAll();

                // Save changes
                userEnum.Update();
            }
            catch (Exception exception)
            {
                _log.TraceException(exception, true);
                throw;
            }
            finally
            {
                // Free resources
                userEnum?.FreeForReuse();
            }
        }

        //------------------------------------------------------------------------------------

        /// <summary>
        /// Update a single user
        /// </summary>
        public void UpdateUser(int userId, string userName, string userDescription, bool isUserEnabled)
        {
            _log.TraceInfo("UpdateUser", $"UserId:          '{userId}'");
            _log.TraceInfo("UpdateUser", $"UserName:        '{userName}'");
            _log.TraceInfo("UpdateUser", $"UserDescription: '{userDescription}'");
            _log.TraceInfo("UpdateUser", $"IsUserEnabled:   '{isUserEnabled}'");

            // Get UserEnum object from LibManager
            var userEnum = _libManager.GetUserEnum();

            try
            {
                // Fill PrimaryCollection with all users
                userEnum.ExecuteIDFilter(userId);
                if (userEnum.PrimaryCollection.Count == 0)
                {
                    throw new ArgumentException($"A user with id {userId} could not be found.");
                }

                // Update user entry
                var userEntry = userEnum.PrimaryCollection[0];
                userEntry.Name = userName;
                userEntry.Comment = userDescription;
                userEntry.Locked = !isUserEnabled;

                // Save all changes
                userEnum.Update();
            }
            catch (Exception exception)
            {
                _log.TraceException(exception, true);
                throw;
            }
            finally
            {
                // Free resources
                userEnum?.FreeForReuse();
            }
        }

        //------------------------------------------------------------------------------

        /// <summary>
        /// Sets a new comment for all IpPbx user
        /// </summary>
        /// <param name="comment">New Comment</param>
        public void EditUserCommentForAllUsers(string comment)
        {
            _log.TraceInfo("EditUserCommentForAllUsers", "Comment: {0}", comment);

            // Get UserEnum object from LibManager
            var userEnum = _libManager.GetUserEnum();

            try
            {
                // Fill PrimaryCollection with all users
                userEnum.ExecuteNameFilter("%", new OrderByList());

                foreach (SWConfigDataClientLib.Proxies.Users.UserEntry userEntry in
                         userEnum.PrimaryCollection)
                {
                    userEntry.Comment = comment;
                }

                // Save all changes
                userEnum.Update();
            }
            catch (Exception exception)
            {
                _log.TraceException(exception, true);
                throw;
            }
            finally
            {
                // Free resources
                userEnum?.FreeForReuse();
            }
        }

        //------------------------------------------------------------------------------

        /// <summary>
        /// Sets a new comment for a single IpPbx user
        /// </summary>
        /// <param name="username">Name of the user</param>
        /// <param name="comment">New comment for the user</param>
        public void EditUserCommentForOneUser(string username, string comment)
        {
            _log.TraceInfo("EditUserCommentForAllUsers",
                "Username: {0} / New Comment: {1}", username, comment);

            // Get UserEnum object from LibManager
            var userEnum = _libManager.GetUserEnum();

            try
            {
                userEnum.ExecuteNameFilter(username, new OrderByList());

                if (userEnum.PrimaryCollection.Count == 0 ||
                    userEnum.PrimaryCollection.Count > 1)
                {
                    throw new Exception("None or more than one user found. Process aborted.");
                }

                foreach (SWConfigDataClientLib.Proxies.Users.UserEntry userEntry in userEnum.PrimaryCollection)
                {
                    userEntry.Comment = comment;
                    userEntry.Update(true);
                }
            }
            catch (Exception exception)
            {
                _log.TraceException(exception, true);
                throw;
            }
            finally
            {
                userEnum?.FreeForReuse();
            }
        }

        #endregion

        #region UserData

        /// <summary>
        /// Updates specific user data for one or multiple users
        /// </summary>
        /// <param name="userIdList">List of user ids</param>
        /// <param name="alwaysOnTop">If TRUE the IpPbx client will be always on top of other windows</param>
        /// <param name="popUpOnIncomingCall">If TRUE the IpPbx client will automatically popup on incoming call</param>
        public void UpdateUserData(List<int> userIdList, bool alwaysOnTop, bool popUpOnIncomingCall)
        {
            // Get a list of all IpPbx users
            var userList = GetUsers();

            // Iterate through all users
            foreach (SWConfigDataClientLib.Proxies.Users.UserEntry userEntry in userList)
            {
                // Skip users that are not in the UserIdList
                if (!userIdList.Contains(userEntry.UserID))
                {
                    continue;
                }

                // Get UserData as byte array
                var userDataByte = userEntry.GetUserConfigData();
                if (userDataByte == null)
                {
                    throw new ArgumentNullException("UserData of user '" + userEntry.Name + "' is null");
                }

                // Convert byte array to useable object
                var userData = new IpPbxBLOB.UserSettings();
                userData.SetBLOB(userDataByte);

                // Update user data
                userData.m_bAlwaysOnTop = alwaysOnTop ? 1 : 0;
                userData.m_bPopUpOnIncomingCall = popUpOnIncomingCall ? 1 : 0;

                // Save user data
                userEntry.SetUserConfigData(userData.GetBLOB());

                // Save user changes
                userEntry.Update(true);
            }

        }

        #endregion

        #region Caller List

        /// <summary>
        /// Returns a UserPhoneCallListEntrySortableCollection for an ippbx user
        /// </summary>
        /// <param name="userId">Id of the ippbx user (use 0 if you
        /// want to get the list of the current user)</param>
        public SWConfigDataClientLib.Proxies.PhoneClient.UserPhoneCallListEntrySortableCollection GetCallerList(int userId)
        {
            _log.TraceInfo("GetCallerList", "({0})", userId);

            // Ensure that a PhoneClientFacade object is available
            if (_phoneClientFacade == null)
            {
                _phoneClientFacade = _libManager.GetPhoneClientFacade();
            }

            try
            {
                return _phoneClientFacade.GetUserPhoneCallList(userId, false).CreateSortableFilterCollection();
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
        }

        //------------------------------------------------------------------------------------

        /// <summary>
        /// Clear the Caller List of an ippbx user
        /// </summary>
        /// <param name="userId">Id of the ippbx user (use 0 if
        /// you want to clear the list of the current user)</param>
        public bool ClearCallerList(int userId)
        {
            _log.TraceInfo("ClearCallerList", "({0})", userId);

            // Ensure that a PhoneClientFacade object is available
            if (_phoneClientFacade == null)
            {
                _phoneClientFacade = _libManager.GetPhoneClientFacade();
            }

            try
            {
                _phoneClientFacade.DeleteUserPhoneCalls(userId);
                return true;
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
        }

        #endregion

        #region Redial List

        /// <summary>
        /// Returns a RedialItemEntrySortableCollection for an ippbx user
        /// </summary>
        /// <param name="userId">Id of the ippbx user (use 0 if
        /// you want to get the list of the current user)</param>
        public SWConfigDataClientLib.Proxies.PhoneClient.RedialItemEntrySortableCollection GetRedialList(int userId)
        {
            _log.TraceInfo("GetRedialList", "({0})", userId);

            // Ensure that a PhoneClientFacade object is available
            if (_phoneClientFacade == null)
            {
                _phoneClientFacade = _libManager.GetPhoneClientFacade();
            }

            try
            {
                return _phoneClientFacade.GetRedialItems(userId).CreateSortableFilterCollection();
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
        }

        //------------------------------------------------------------------------------------

        /// <summary>
        /// Clear the Redial List of an ippbx user
        /// </summary>
        /// <param name="userId">Id of the ippbx user (use 0 if
        /// you want to clear the list of the current user)</param>
        public bool ClearRedialList(int userId)
        {
            _log.TraceInfo("ClearRedialList", "({0})", userId);

            // Ensure that a PhoneClientFacade object is available
            if (_phoneClientFacade == null)
            {
                _phoneClientFacade = _libManager.GetPhoneClientFacade();
            }

            try
            {
                _phoneClientFacade.DeleteRedialItems(userId);
                return true;
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
        }

        #endregion

        #region Active Calls List

        /// <summary>
        /// Returns a list of the active calls
        /// </summary>
        public ActiveCallEntrySortableCollection GetActiveCalls()
        {
            _log.TraceInfo("GetActiveCalls", "()");

            // Ensure that a AdminFacade object is available
            if (_adminFacade == null)
            {
                _adminFacade = _libManager.GetAdminFacade();
            }

            try
            {
                return _adminFacade.GetActiveCallsView().CreateSortableFilterCollection();
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
        }

        #endregion

        #region Phonebook

        public void ImportPersonalPhonebook(UserAdminView1Entry userEntry, List<UserPhoneBookEntry> newPersonalPhonebookData, bool clearUserPhonebook)
        {
            _log.TraceInfo("ImportPersonalPhonebook", "User '{0}' ({1}) / ClearUserPhonebook: {2}",
                userEntry.Name, userEntry.UserID, clearUserPhonebook);

            var userPhoneBookEnum = _libManager.GetUserPhoneBookEnum();

            try
            {
                userPhoneBookEnum.ExecuteFilterAll(userEntry.UserID, new OrderByList());

                if (clearUserPhonebook)
                {
                    userPhoneBookEnum.PrimaryCollection.RemoveAll();
                    userPhoneBookEnum.Update();
                }

                foreach (var pbEntry in newPersonalPhonebookData)
                {
                    // Because of the unique entryId we have to create a new UserPhoneBookEntry
                    // Furthermore the UserID value must be updated
                    var newEntry = new UserPhoneBookEntry();
                    newEntry.ApplyData(pbEntry);
                    newEntry.UserID = userEntry.UserID;

                    userPhoneBookEnum.PrimaryCollection.Add(newEntry);
                }

                userPhoneBookEnum.Update();
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
            finally
            {
                // Free resources
                userPhoneBookEnum?.FreeForReuse();
            }
        }

        //------------------------------------------------------------------------------------

        public void ClearPersonalPhonebook(UserAdminView1Entry userEntry)
        {
            _log.TraceInfo("ClearPersonalPhonebook", "User '{0}' ({1})", userEntry.Name, userEntry.UserID);

            var userPhoneBookEnum = _libManager.GetUserPhoneBookEnum();

            try
            {
                userPhoneBookEnum.ExecuteFilterAll(userEntry.UserID, new OrderByList());
                userPhoneBookEnum.PrimaryCollection.RemoveAll();
                userPhoneBookEnum.Update();
            }
            catch (Exception ex)
            {
                _log.TraceException(ex, true);
                throw;
            }
            finally
            {
                // Free resources
                userPhoneBookEnum?.FreeForReuse();
            }
        }

        #endregion        
    }
}